import React, { useState, useRef, useEffect } from 'react';
import { Bot, X, Send, Sparkles } from 'lucide-react';
import { sendMessageToGemini } from '../services/geminiService';
import { ChatMessage } from '../types';

export const GeminiAssistant: React.FC = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [input, setInput] = useState('');
  const [messages, setMessages] = useState<ChatMessage[]>([
    { id: '0', role: 'model', text: '您好！我是您的智慧小幫手阿德。有什麼我可以幫您的嗎？我可以幫您查公車、推薦課程，或者陪您聊天喔！' }
  ]);
  const [isLoading, setIsLoading] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  useEffect(() => {
    scrollToBottom();
  }, [messages, isOpen]);

  const handleSend = async () => {
    if (!input.trim() || isLoading) return;

    const userMsg: ChatMessage = { id: Date.now().toString(), role: 'user', text: input };
    setMessages(prev => [...prev, userMsg]);
    setInput('');
    setIsLoading(true);

    try {
      const responseText = await sendMessageToGemini(userMsg.text);
      const aiMsg: ChatMessage = { id: (Date.now() + 1).toString(), role: 'model', text: responseText };
      setMessages(prev => [...prev, aiMsg]);
    } catch (error) {
      const errorMsg: ChatMessage = { id: (Date.now() + 1).toString(), role: 'model', text: '抱歉，系統忙碌中。', isError: true };
      setMessages(prev => [...prev, errorMsg]);
    } finally {
      setIsLoading(false);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') handleSend();
  };

  return (
    <>
      {/* Floating Button */}
      {!isOpen && (
        <button
          onClick={() => setIsOpen(true)}
          className="absolute bottom-20 right-4 bg-secondary text-white p-4 rounded-full shadow-lg hover:bg-orange-600 transition-transform transform hover:scale-105 z-30 flex items-center gap-2"
        >
          <Bot size={28} />
          <span className="font-bold">問阿德</span>
        </button>
      )}

      {/* Chat Window Overlay */}
      {isOpen && (
        <div className="absolute inset-0 z-50 bg-black/50 flex items-end justify-center">
          <div className="bg-white w-full h-[85%] rounded-t-2xl flex flex-col shadow-2xl animate-slide-up">
            
            {/* Header */}
            <div className="bg-gradient-to-r from-secondary to-orange-400 p-4 rounded-t-2xl flex justify-between items-center text-white">
              <div className="flex items-center gap-2">
                <div className="bg-white/20 p-2 rounded-full">
                    <Sparkles size={20} />
                </div>
                <h2 className="font-bold text-lg">智慧助手 阿德</h2>
              </div>
              <button onClick={() => setIsOpen(false)} className="hover:bg-white/20 p-1 rounded">
                <X size={24} />
              </button>
            </div>

            {/* Messages Area */}
            <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-gray-50">
              {messages.map((msg) => (
                <div key={msg.id} className={`flex ${msg.role === 'user' ? 'justify-end' : 'justify-start'}`}>
                  <div
                    className={`max-w-[80%] p-3 rounded-2xl text-lg leading-relaxed ${
                      msg.role === 'user'
                        ? 'bg-primary text-white rounded-br-none'
                        : 'bg-white text-gray-800 border border-gray-200 rounded-bl-none shadow-sm'
                    } ${msg.isError ? 'bg-red-100 text-red-600' : ''}`}
                  >
                    {msg.text}
                  </div>
                </div>
              ))}
              {isLoading && (
                <div className="flex justify-start">
                  <div className="bg-gray-200 p-3 rounded-2xl rounded-bl-none animate-pulse">
                    正在思考中...
                  </div>
                </div>
              )}
              <div ref={messagesEndRef} />
            </div>

            {/* Input Area */}
            <div className="p-4 bg-white border-t border-gray-200 flex gap-2">
              <input
                type="text"
                value={input}
                onChange={(e) => setInput(e.target.value)}
                onKeyDown={handleKeyPress}
                placeholder="請輸入您的問題..."
                className="flex-1 border border-gray-300 rounded-full px-4 py-3 text-lg focus:outline-none focus:border-secondary focus:ring-2 focus:ring-orange-200"
              />
              <button
                onClick={handleSend}
                disabled={isLoading}
                className="bg-secondary text-white p-3 rounded-full hover:bg-orange-600 disabled:opacity-50"
              >
                <Send size={24} />
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  );
};
