// DOM元素 - 安全取得，避免在其他頁面產生錯誤
const hamburgerBtn = document.getElementById('hamburger');
const sideMenu = document.getElementById('sideMenu');
const menuOverlay = document.getElementById('menuOverlay');
const closeMenuBtn = document.getElementById('closeMenu');
const carousel = document.getElementById('carousel');

// 輪播相關變數
let currentSlideIndex = 0;
const slides = document.querySelectorAll('.carousel-slide');
const dots = document.querySelectorAll('.dot');
let slideInterval;

// 初始化
document.addEventListener('DOMContentLoaded', function() {
    // 只在元素存在時執行對應的初始化
    if (hamburgerBtn && sideMenu) {
        initMenu();
    }
    if (carousel) {
        initCarousel();
    }
    // 這些功能在所有頁面都可能需要
    initHomeButton();
    initNotificationButton();
});

// 初始化選單功能
function initMenu() {
    if (!hamburgerBtn || !sideMenu || !closeMenuBtn || !menuOverlay) {
        return; // 如果必要元素不存在，直接返回
    }
    
    // 開啟選單
    hamburgerBtn.addEventListener('click', function() {
        toggleMenu(true);
    });

    // 關閉選單
    closeMenuBtn.addEventListener('click', function() {
        toggleMenu(false);
    });

    // 點擊遮罩關閉選單
    menuOverlay.addEventListener('click', function() {
        toggleMenu(false);
    });

    // ESC 鍵關閉選單
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && sideMenu.classList.contains('active')) {
            toggleMenu(false);
        }
    });

    // 選單項目點擊事件
    const menuItems = document.querySelectorAll('.menu-item');
    menuItems.forEach(item => {
        item.addEventListener('click', function(e) {
            const text = this.textContent;
            const href = this.getAttribute('href');
            
            // GO食有實際連結，不需要阻止預設行為
            if (href && href !== '#') {
                toggleMenu(false);
                return;
            }
            
            e.preventDefault();
            
            // 簡單的頁面跳轉提示（實際應用中會連接到對應頁面）
            if (text !== '回首頁') {
                showToast(`即將跳轉到${text}頁面`);
            } else {
                showToast('已在首頁');
            }
            
            toggleMenu(false);
        });
    });
}

// 切換選單顯示
function toggleMenu(show) {
    if (show) {
        sideMenu.classList.add('active');
        hamburgerBtn.classList.add('active');
        document.body.style.overflow = 'hidden'; // 防止背景滾動
    } else {
        sideMenu.classList.remove('active');
        hamburgerBtn.classList.remove('active');
        document.body.style.overflow = '';
    }
}

// 初始化輪播功能
function initCarousel() {
    if (!carousel || slides.length === 0) return;

    // 開始自動播放
    startAutoPlay();

    // 滑鼠進入暫停，離開繼續
    const carouselContainer = document.querySelector('.carousel-container');
    if (!carouselContainer) return;
    
    carouselContainer.addEventListener('mouseenter', stopAutoPlay);
    carouselContainer.addEventListener('mouseleave', startAutoPlay);

    // 觸控支援
    let touchStartX = 0;
    let touchEndX = 0;

    carouselContainer.addEventListener('touchstart', function(e) {
        touchStartX = e.changedTouches[0].screenX;
        stopAutoPlay();
    });

    carouselContainer.addEventListener('touchend', function(e) {
        touchEndX = e.changedTouches[0].screenX;
        handleSwipe();
        startAutoPlay();
    });

    function handleSwipe() {
        const swipeThreshold = 50;
        const diff = touchStartX - touchEndX;

        if (Math.abs(diff) > swipeThreshold) {
            if (diff > 0) {
                changeSlide(1); // 向左滑，下一張
            } else {
                changeSlide(-1); // 向右滑，上一張
            }
        }
    }

    // 鍵盤支援
    document.addEventListener('keydown', function(e) {
        if (e.key === 'ArrowLeft') {
            changeSlide(-1);
        } else if (e.key === 'ArrowRight') {
            changeSlide(1);
        }
    });
}

// 切換輪播圖片
function changeSlide(direction) {
    if (!slides || slides.length === 0 || !dots || dots.length === 0) return;
    
    // 移除當前活動狀態
    if (slides[currentSlideIndex]) {
        slides[currentSlideIndex].classList.remove('active');
    }
    if (dots[currentSlideIndex]) {
        dots[currentSlideIndex].classList.remove('active');
    }

    // 計算新的索引
    currentSlideIndex += direction;

    // 循環處理
    if (currentSlideIndex >= slides.length) {
        currentSlideIndex = 0;
    } else if (currentSlideIndex < 0) {
        currentSlideIndex = slides.length - 1;
    }

    // 添加新的活動狀態
    if (slides[currentSlideIndex]) {
        slides[currentSlideIndex].classList.add('active');
    }
    if (dots[currentSlideIndex]) {
        dots[currentSlideIndex].classList.add('active');
    }

    // 重啟自動播放
    stopAutoPlay();
    startAutoPlay();
}

// 直接跳轉到指定幻燈片
function currentSlide(index) {
    if (!slides || slides.length === 0 || !dots || dots.length === 0) return;
    
    const targetIndex = index - 1; // 轉換為 0 基底索引
    
    if (targetIndex < 0 || targetIndex >= slides.length || targetIndex === currentSlideIndex) return;

    // 移除當前活動狀態
    if (slides[currentSlideIndex]) {
        slides[currentSlideIndex].classList.remove('active');
    }
    if (dots[currentSlideIndex]) {
        dots[currentSlideIndex].classList.remove('active');
    }

    // 設定新的索引和狀態
    currentSlideIndex = targetIndex;
    if (slides[currentSlideIndex]) {
        slides[currentSlideIndex].classList.add('active');
    }
    if (dots[currentSlideIndex]) {
        dots[currentSlideIndex].classList.add('active');
    }

    // 重啟自動播放
    stopAutoPlay();
    startAutoPlay();
}

// 開始自動播放
function startAutoPlay() {
    if (!slides || slides.length === 0) return;
    
    slideInterval = setInterval(() => {
        changeSlide(1);
    }, 4000); // 4秒切換一次
}

// 停止自動播放
function stopAutoPlay() {
    if (slideInterval) {
        clearInterval(slideInterval);
        slideInterval = null;
    }
}

// 初始化首頁按鈕
function initHomeButton() {
    const homeBtn = document.querySelector('.home-btn');
    if (homeBtn) {
        homeBtn.addEventListener('click', function() {
            // 滾動到頂部
            window.scrollTo({ top: 0, behavior: 'smooth' });
            showToast('已回到首頁頂部');
        });
    }
}

// 初始化通知按鈕
function initNotificationButton() {
    const notificationBtn = document.querySelector('.notification-btn');
    if (notificationBtn) {
        notificationBtn.addEventListener('click', function() {
            showToast('目前沒有新通知');
            
            // 添加小動畫效果
            this.style.transform = 'scale(0.95)';
            setTimeout(() => {
                this.style.transform = 'scale(1)';
            }, 150);
        });
    }
}

// 快速入口點擊事件
document.addEventListener('DOMContentLoaded', function() {
    const quickLinks = document.querySelectorAll('.quick-link-item');
    quickLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            const text = this.querySelector('span').textContent;
            const href = this.getAttribute('href');
            
            // GO食有實際連結，不需要阻止預設行為
            if (href && href !== '#') {
                // 添加點擊動畫
                this.style.transform = 'translateY(-3px) scale(0.95)';
                setTimeout(() => {
                    this.style.transform = '';
                }, 200);
                return;
            }
            
            e.preventDefault();
            showToast(`即將跳轉到${text}頁面`);
            
            // 添加點擊動畫
            this.style.transform = 'translateY(-3px) scale(0.95)';
            setTimeout(() => {
                this.style.transform = '';
            }, 200);
        });
    });
});

// Toast 提示功能
function showToast(message, duration = 3000) {
    // 移除現有的 toast
    const existingToast = document.querySelector('.toast');
    if (existingToast) {
        existingToast.remove();
    }

    // 創建 toast 元素
    const toast = document.createElement('div');
    toast.className = 'toast';
    toast.textContent = message;
    
    // 添加 toast 樣式
    Object.assign(toast.style, {
        position: 'fixed',
        bottom: '30px',
        left: '50%',
        transform: 'translateX(-50%)',
        backgroundColor: 'var(--primary-color)',
        color: 'white',
        padding: '16px 24px',
        borderRadius: '8px',
        fontSize: '16px',
        fontWeight: '500',
        boxShadow: '0 4px 12px rgba(0,0,0,0.3)',
        zIndex: '10000',
        opacity: '0',
        transition: 'all 0.3s ease',
        maxWidth: '90%',
        textAlign: 'center'
    });

    document.body.appendChild(toast);

    // 顯示動畫
    setTimeout(() => {
        toast.style.opacity = '1';
        toast.style.transform = 'translateX(-50%) translateY(-10px)';
    }, 100);

    // 自動隱藏
    setTimeout(() => {
        toast.style.opacity = '0';
        toast.style.transform = 'translateX(-50%) translateY(20px)';
        setTimeout(() => {
            if (toast.parentNode) {
                toast.remove();
            }
        }, 300);
    }, duration);
}

// 頁面載入完成後的初始化
window.addEventListener('load', function() {
    // 添加載入完成的淡入效果
    document.body.style.opacity = '0';
    document.body.style.transition = 'opacity 0.5s ease';
    
    setTimeout(() => {
        document.body.style.opacity = '1';
    }, 100);
    
    // 只在首頁顯示歡迎提示
    if (window.location.pathname.endsWith('index.html') || window.location.pathname === '/' || document.title.includes('首頁')) {
        setTimeout(() => {
            showToast('歡迎來到金華社區！', 2000);
        }, 1000);
    }
});

// 響應式處理
function handleResize() {
    // 在調整視窗大小時關閉選單（避免佈局問題）
    if (window.innerWidth > 768 && sideMenu.classList.contains('active')) {
        toggleMenu(false);
    }
}

window.addEventListener('resize', handleResize);

// 防止頁面意外刷新時的資料保存
window.addEventListener('beforeunload', function() {
    // 清理定時器
    stopAutoPlay();
});

// 錯誤處理 - 僅在首頁顯示通用錯誤訊息
window.addEventListener('error', function(e) {
    console.error('頁面發生錯誤:', e.error);
    
    // 只在首頁顯示通用錯誤訊息，其他頁面有自己的錯誤處理
    const isHomePage = window.location.pathname.endsWith('index.html') || 
                      window.location.pathname === '/' || 
                      document.title.includes('金華社區');
    
    if (isHomePage) {
        showToast('頁面載入時發生錯誤，請重新整理頁面', 5000);
    }
});

// 提供全域函數（供 HTML 中的 onclick 使用）
window.changeSlide = changeSlide;
window.currentSlide = currentSlide;
