// 公告詳細頁面 JavaScript 功能

// 模擬公告資料（與 goxiang-script.js 相同）
const mockAnnouncementDetails = {
    '1': {
        id: '1',
        type: 'emergency',
        title: '緊急通知：社區停水公告',
        content: `因水管維修工程，本社區將於明日上午8:00-下午6:00進行停水作業，請各位里民提前做好儲水準備。
        
停水範圍：整個社區
停水時間：2025年1月18日 08:00 - 18:00
預計復水時間：2025年1月18日 18:30

如有緊急用水需求，可聯繫管理室。造成不便敬請見諒。

聯絡電話：02-1234-5678
管理室位置：社區大門旁

感謝各位里民的配合與諒解。`,
        keywords: ['停水', '維修', '緊急'],
        publishedAt: '2025-01-17T10:30:00Z',
        pinnedUntil: '2025-01-20T23:59:59Z',
        photos: ['images/1.jpg', 'images/2.jpg']
    },
    '2': {
        id: '2',
        type: 'event',
        title: '社區健康講座',
        content: `邀請專業醫師分享健康養生知識，歡迎全體里民踴躍參加。
        
本次講座邀請到台北醫學大學附設醫院的專業醫師，將針對以下主題進行分享：
• 銀髮族健康管理
• 慢性病預防與保健
• 營養飲食建議
• 運動與健康

現場提供免費健康檢測服務，包含：
• 血壓測量
• 血糖檢測
• 體重體脂測量
• 健康諮詢

活動完全免費，歡迎攜家帶眷一同參與！`,
        keywords: ['健康', '講座', '醫師', '免費'],
        publishedAt: '2025-01-16T09:15:00Z',
        eventMeta: {
            location: '社區活動中心',
            date: '2025-01-25',
            timeSlot: '14:00'
        },
        photos: ['images/2.jpg']
    },
    '3': {
        id: '3',
        type: 'general',
        title: '社區環境清潔日',
        content: `為維護社區環境品質，將定期舉辦清潔日活動，歡迎里民共同參與，一起打造美好家園。
        
活動內容：
• 公共區域清潔整理
• 花園植栽維護
• 垃圾分類宣導
• 環境美化工作

參與方式：
• 自備清潔用具
• 穿著輕便服裝
• 戴帽子及手套
• 攜帶飲用水

每位參與者將獲得：
• 環保紀念品一份
• 免費點心飲料
• 志工服務時數認證

讓我們一起為美好的社區環境盡一份心力！`,
        keywords: ['清潔', '環境', '參與', '志工'],
        publishedAt: '2025-01-15T16:20:00Z',
        photos: ['images/3.png']
    },
    '4': {
        id: '4',
        type: 'event',
        title: '新春團拜活動',
        content: `農曆新年將至，社區將舉辦團拜活動，共同迎接新年到來。
        
活動流程：
18:30-19:00 開場致詞
19:00-19:30 精彩表演
19:30-20:00 聚餐時間
20:00-20:30 摸彩活動
20:30-21:00 新年祝福

表演節目：
• 社區合唱團演出
• 太極拳表演
• 兒童舞蹈
• 傳統樂器演奏

摸彩獎品：
頭獎：液晶電視 32吋
二獎：微波爐
三獎：健康禮盒
參加獎：紅包袋

請各位里民踴躍參加，一起歡度新春！`,
        keywords: ['新春', '團拜', '活動', '摸彩'],
        publishedAt: '2025-01-14T11:45:00Z',
        eventMeta: {
            location: '社區廣場',
            date: '2025-01-30',
            timeSlot: '18:30'
        },
        photos: ['images/1.jpg', 'images/2.jpg']
    }
};

// 頁面狀態
let currentAnnouncement = null;
let currentPhotoIndex = 0;
let photos = [];

// DOM 元素
let announcementMain;
let loadingContainer;
let announcementContent;
let errorContainer;
let photoViewerModal;
let photoViewerImage;
let photoIndicators;
let photoPrev;
let photoNext;

// 頁面初始化
document.addEventListener('DOMContentLoaded', function() {
    initializeElements();
    loadAnnouncement();
    
    console.log('公告詳細頁面初始化完成');
});

// 初始化 DOM 元素
function initializeElements() {
    announcementMain = document.getElementById('announcementMain');
    loadingContainer = document.getElementById('loadingContainer');
    announcementContent = document.getElementById('announcementContent');
    errorContainer = document.getElementById('errorContainer');
    photoViewerModal = document.getElementById('photoViewerModal');
    photoViewerImage = document.getElementById('photoViewerImage');
    photoIndicators = document.getElementById('photoIndicators');
    photoPrev = document.getElementById('photoPrev');
    photoNext = document.getElementById('photoNext');
}

// 載入公告詳細
function loadAnnouncement() {
    // 顯示載入狀態
    showLoading();
    
    // 從 URL 參數獲取公告 ID
    const urlParams = new URLSearchParams(window.location.search);
    const announcementId = urlParams.get('id');
    
    if (!announcementId) {
        showError('無效的公告 ID');
        return;
    }
    
    // 模擬 API 調用延遲
    setTimeout(() => {
        const announcement = mockAnnouncementDetails[announcementId];
        
        if (!announcement) {
            showError('找不到指定的公告');
            return;
        }
        
        currentAnnouncement = announcement;
        photos = announcement.photos || [];
        renderAnnouncement(announcement);
        showContent();
    }, 800);
}

// 顯示載入狀態
function showLoading() {
    loadingContainer.style.display = 'flex';
    announcementContent.style.display = 'none';
    errorContainer.style.display = 'none';
}

// 顯示錯誤狀態
function showError(message) {
    loadingContainer.style.display = 'none';
    announcementContent.style.display = 'none';
    errorContainer.style.display = 'flex';
    
    const errorMessage = errorContainer.querySelector('.error-message');
    if (errorMessage) {
        errorMessage.textContent = message;
    }
}

// 顯示內容
function showContent() {
    loadingContainer.style.display = 'none';
    announcementContent.style.display = 'block';
    errorContainer.style.display = 'none';
}

// 渲染公告內容
function renderAnnouncement(announcement) {
    if (!announcementContent) return;
    
    // 設定公告類型樣式
    announcementContent.className = `announcement-content ${announcement.type}`;
    
    // 生成內容 HTML
    const contentHTML = generateAnnouncementHTML(announcement);
    announcementContent.innerHTML = contentHTML;
    
    // 添加照片點擊事件
    addPhotoClickEvents();
}

// 生成公告 HTML
function generateAnnouncementHTML(announcement) {
    const typeText = {
        'general': '一般公告',
        'emergency': '緊急公告',
        'event': '活動公告'
    };
    
    // 主圖片（如果有照片）
    const heroImageStyle = announcement.photos && announcement.photos.length > 0 
        ? `style="background-image: url('${announcement.photos[0]}');"` 
        : '';
    const heroClass = announcement.photos && announcement.photos.length > 0 ? 'has-image' : '';
    
    // 活動資訊區塊
    const eventInfoHTML = announcement.eventMeta ? `
        <section class="event-info-section">
            <div class="event-info-grid">
                <div class="event-info-item">
                    <div class="event-info-icon">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M12 2C8.13 2 5 5.13 5 9C5 14.25 12 22 12 22S19 14.25 19 9C19 5.13 15.87 2 12 2ZM12 11.5C10.62 11.5 9.5 10.38 9.5 9S10.62 6.5 12 6.5S14.5 7.62 14.5 9S13.38 11.5 12 11.5Z" fill="currentColor"/>
                        </svg>
                    </div>
                    <div class="event-info-label">地點</div>
                    <div class="event-info-value">${announcement.eventMeta.location}</div>
                </div>
                <div class="event-info-item">
                    <div class="event-info-icon">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M9 11H7V13H9V11ZM13 11H11V13H13V11ZM17 11H15V13H17V11ZM19 4H18V2H16V4H8V2H6V4H5C3.89 4 3.01 4.9 3.01 6L3 20C3 21.1 3.89 22 5 22H19C20.1 22 21 21.1 21 20V6C21 4.9 20.1 4 19 4ZM19 20H5V9H19V20Z" fill="currentColor"/>
                        </svg>
                    </div>
                    <div class="event-info-label">日期</div>
                    <div class="event-info-value">${formatEventDate(announcement.eventMeta.date)}</div>
                </div>
                <div class="event-info-item">
                    <div class="event-info-icon">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M16.2,16.2L11,13V7H12.5V12.2L17,14.7L16.2,16.2Z" fill="currentColor"/>
                        </svg>
                    </div>
                    <div class="event-info-label">時間</div>
                    <div class="event-info-value">${announcement.eventMeta.timeSlot}</div>
                </div>
            </div>
        </section>
    ` : '';
    
    // 關鍵字標籤
    const keywordsHTML = announcement.keywords && announcement.keywords.length > 0 ? `
        <section class="content-section">
            <h2 class="section-title">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M17.63 5.84C17.27 5.33 16.67 5 16 5L5 5.01C3.9 5.01 3 5.9 3 7V17C3 18.1 3.9 19 5 19H16C16.67 19 17.27 18.67 17.63 18.16L22 12L17.63 5.84ZM16 17H5V7H16L20.55 12L16 17Z" fill="currentColor"/>
                </svg>
                相關標籤
            </h2>
            <div class="keywords-container">
                ${announcement.keywords.map(keyword => 
                    `<span class="keyword-tag">${keyword}</span>`
                ).join('')}
            </div>
        </section>
    ` : '';
    
    // 照片區塊
    const photosHTML = announcement.photos && announcement.photos.length > 0 ? `
        <section class="content-section">
            <h2 class="section-title">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M21 19V5C21 3.9 20.1 3 19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19ZM8.5 13.5L11 16.51L14.5 12L19 18H5L8.5 13.5Z" fill="currentColor"/>
                </svg>
                相關圖片 (${announcement.photos.length})
            </h2>
            <div class="photos-grid">
                ${announcement.photos.map((photo, index) => `
                    <div class="photo-item" data-photo-index="${index}">
                        <img src="${photo}" alt="公告圖片 ${index + 1}">
                    </div>
                `).join('')}
            </div>
        </section>
    ` : '';
    
    return `
        <header class="announcement-hero ${heroClass}" ${heroImageStyle}>
            <div class="hero-overlay"></div>
            <div class="hero-content">
                <span class="announcement-type">${typeText[announcement.type]}</span>
                <h1 class="announcement-title">${announcement.title}</h1>
                <div class="announcement-meta">
                    <div class="meta-item">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M9 11H7V13H9V11ZM13 11H11V13H13V11ZM17 11H15V13H17V11ZM19 4H18V2H16V4H8V2H6V4H5C3.89 4 3.01 4.9 3.01 6L3 20C3 21.1 3.89 22 5 22H19C20.1 22 21 21.1 21 20V6C21 4.9 20.1 4 19 4ZM19 20H5V9H19V20Z" fill="currentColor"/>
                        </svg>
                        ${formatDate(announcement.publishedAt)}
                    </div>
                    ${announcement.pinnedUntil ? `
                        <div class="meta-item">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M16,12V4H17V2H7V4H8V12L6,14V16H11.2V22H12.8V16H18V14L16,12Z" fill="currentColor"/>
                            </svg>
                            置頂至 ${formatDate(announcement.pinnedUntil)}
                        </div>
                    ` : ''}
                </div>
            </div>
        </header>
        
        ${eventInfoHTML}
        
        <section class="announcement-body">
            <section class="content-section">
                <h2 class="section-title">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M14 2H6C4.9 2 4.01 2.9 4.01 4L4 20C4 21.1 4.89 22 6 22H18C19.1 22 20 21.1 20 20V8L14 2ZM18 20H6V4H13V9H18V20Z" fill="currentColor"/>
                    </svg>
                    公告內容
                </h2>
                <div class="content-text">${announcement.content}</div>
            </section>
            
            ${keywordsHTML}
            ${photosHTML}
        </section>
    `;
}

// 添加照片點擊事件
function addPhotoClickEvents() {
    const photoItems = document.querySelectorAll('.photo-item');
    photoItems.forEach(item => {
        item.addEventListener('click', () => {
            const photoIndex = parseInt(item.dataset.photoIndex);
            openPhotoViewer(photoIndex);
        });
    });
}

// 開啟照片檢視器
function openPhotoViewer(photoIndex = 0) {
    if (!photos || photos.length === 0) return;
    
    currentPhotoIndex = photoIndex;
    updatePhotoViewer();
    showPhotoViewer();
    generatePhotoIndicators();
    updatePhotoNavigation();
}

// 顯示照片檢視器
function showPhotoViewer() {
    if (photoViewerModal) {
        photoViewerModal.style.display = 'flex';
        setTimeout(() => {
            photoViewerModal.classList.add('show');
        }, 10);
        
        // 防止背景滾動
        document.body.style.overflow = 'hidden';
    }
}

// 關閉照片檢視器
function closePhotoViewer() {
    if (photoViewerModal) {
        photoViewerModal.classList.remove('show');
        setTimeout(() => {
            photoViewerModal.style.display = 'none';
        }, 300);
        
        // 恢復背景滾動
        document.body.style.overflow = '';
    }
}

// 更新照片檢視器
function updatePhotoViewer() {
    if (photoViewerImage && photos[currentPhotoIndex]) {
        photoViewerImage.src = photos[currentPhotoIndex];
        photoViewerImage.alt = `公告圖片 ${currentPhotoIndex + 1}`;
    }
    
    updatePhotoIndicators();
}

// 生成照片指示器
function generatePhotoIndicators() {
    if (!photoIndicators || photos.length <= 1) {
        if (photoIndicators) photoIndicators.style.display = 'none';
        return;
    }
    
    photoIndicators.style.display = 'flex';
    photoIndicators.innerHTML = photos.map((_, index) => 
        `<button class="photo-indicator ${index === currentPhotoIndex ? 'active' : ''}" 
                 onclick="goToPhoto(${index})"></button>`
    ).join('');
}

// 更新照片指示器
function updatePhotoIndicators() {
    const indicators = photoIndicators?.querySelectorAll('.photo-indicator');
    if (indicators) {
        indicators.forEach((indicator, index) => {
            indicator.classList.toggle('active', index === currentPhotoIndex);
        });
    }
}

// 更新照片導航按鈕
function updatePhotoNavigation() {
    if (photoPrev) {
        photoPrev.style.display = photos.length > 1 ? 'block' : 'none';
    }
    if (photoNext) {
        photoNext.style.display = photos.length > 1 ? 'block' : 'none';
    }
}

// 前往指定照片
function goToPhoto(photoIndex) {
    if (photoIndex >= 0 && photoIndex < photos.length) {
        currentPhotoIndex = photoIndex;
        updatePhotoViewer();
    }
}

// 顯示上一張照片
function showPrevPhoto() {
    const prevIndex = currentPhotoIndex === 0 ? photos.length - 1 : currentPhotoIndex - 1;
    goToPhoto(prevIndex);
}

// 顯示下一張照片
function showNextPhoto() {
    const nextIndex = currentPhotoIndex === photos.length - 1 ? 0 : currentPhotoIndex + 1;
    goToPhoto(nextIndex);
}

// 分享公告
function shareAnnouncement() {
    if (!currentAnnouncement) return;
    
    const shareData = {
        title: `${currentAnnouncement.title} - 金華社區`,
        text: currentAnnouncement.content.substring(0, 100) + '...',
        url: window.location.href
    };
    
    // 檢查是否支援 Web Share API
    if (navigator.share) {
        navigator.share(shareData)
            .then(() => {
                showToast('分享成功', 'success');
            })
            .catch(err => {
                console.log('分享失敗:', err);
                fallbackShare();
            });
    } else {
        fallbackShare();
    }
}

// 備用分享方式
function fallbackShare() {
    // 複製連結到剪貼簿
    if (navigator.clipboard) {
        navigator.clipboard.writeText(window.location.href)
            .then(() => {
                showToast('連結已複製到剪貼簿', 'success');
            })
            .catch(err => {
                console.log('複製失敗:', err);
                showToast('分享功能暫時無法使用', 'error');
            });
    } else {
        showToast('分享功能暫時無法使用', 'error');
    }
}

// 返回上一頁
function goBack() {
    // 檢查是否有 referrer，如果有則返回上一頁，否則前往 GO想 首頁
    if (document.referrer && document.referrer.includes(window.location.origin)) {
        window.history.back();
    } else {
        window.location.href = 'goxiang.html';
    }
}

// 顯示 Toast 通知
function showToast(message, type = 'success') {
    const toast = document.getElementById('toast');
    const toastMessage = toast?.querySelector('.toast-message');
    const toastIcon = toast?.querySelector('.toast-icon');
    
    if (toast && toastMessage) {
        toastMessage.textContent = message;
        
        // 重置類別
        toast.classList.remove('error', 'info', 'success');
        
        // 設定圖示和樣式
        switch (type) {
            case 'success':
                toastIcon.textContent = '✓';
                toast.classList.add('success');
                break;
            case 'error':
                toastIcon.textContent = '✕';
                toast.classList.add('error');
                break;
            case 'info':
                toastIcon.textContent = 'i';
                toast.classList.add('info');
                break;
        }
        
        toast.classList.add('show');
        
        setTimeout(() => {
            toast.classList.remove('show');
        }, 3000);
    }
}

// 格式化日期
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('zh-TW', {
        year: 'numeric',
        month: 'long',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// 格式化活動日期
function formatEventDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('zh-TW', {
        month: 'numeric',
        day: 'numeric',
        weekday: 'short'
    });
}

// 鍵盤事件處理
document.addEventListener('keydown', function(e) {
    if (photoViewerModal?.style.display === 'flex') {
        switch (e.key) {
            case 'Escape':
                closePhotoViewer();
                break;
            case 'ArrowLeft':
                e.preventDefault();
                showPrevPhoto();
                break;
            case 'ArrowRight':
                e.preventDefault();
                showNextPhoto();
                break;
        }
    }
});

// 點擊模態框外部關閉
document.addEventListener('click', function(e) {
    if (e.target === photoViewerModal) {
        closePhotoViewer();
    }
});

// 觸控手勢支援（簡單版本）
let touchStartX = 0;
let touchEndX = 0;

document.addEventListener('touchstart', function(e) {
    if (photoViewerModal?.style.display === 'flex') {
        touchStartX = e.changedTouches[0].screenX;
    }
});

document.addEventListener('touchend', function(e) {
    if (photoViewerModal?.style.display === 'flex') {
        touchEndX = e.changedTouches[0].screenX;
        handleSwipe();
    }
});

function handleSwipe() {
    const swipeThreshold = 50;
    const swipeDistance = touchEndX - touchStartX;
    
    if (Math.abs(swipeDistance) > swipeThreshold) {
        if (swipeDistance > 0) {
            // 向右滑動 - 上一張
            showPrevPhoto();
        } else {
            // 向左滑動 - 下一張
            showNextPhoto();
        }
    }
}

// 導出主要功能（供測試使用）
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        loadAnnouncement,
        shareAnnouncement,
        showToast,
        formatDate
    };
}
